/*!
* @program
* @brief Program and routines for controlling the tree view behaviour.
* @author Daniel Enache
* @version 2.0
* @date 20.08.2017
*
* @history  DATE         VERS.  AUTHOR            DESCRIPTION
* @history  13.04.2014    1.0   "Daniel Enache"   Initial implementation.
* @history  20.08.2017    2.0   "Daniel Enache"   - Toogle expand / collapse all added.
*                                                 - Search implemented.
*/
 
/*!******************************************************
* @progsection Basic functionality. 
********************************************************/

/*! @variable g_ListMargin Global variable for tree list margin. */
var g_ListMargin = "0.5em";

/*!
* @function "initTree(oList, l_Margin)"
* @brief Initialization of the Tree.
* @param oList    Parent node of the list.
* @param l_Margin Left margin to be set (string).
*/
function initTree(oList, l_Margin)
{
     // set global variable
     g_ListMargin = l_Margin;

     // collapses the tree
     oList.style.marginLeft = g_ListMargin;
     oList.style.listStyleImage = "url(TreeCollapsed.gif)";
     for (var i=0; i < oList.getElementsByTagName("ul").length; i++)
     {
         oList.getElementsByTagName("ul")[i].style.display = "none";
         oList.getElementsByTagName("ul")[i].style.marginLeft = g_ListMargin;
         oList.getElementsByTagName("ul")[i].style.listStyleImage = "url(TreeCollapsed.gif)";
     }

     // adds the click-event handler to the list items
     if (oList.addEventListener)
     {
         oList.addEventListener("click", onBranchClick, false);
     }
     else if (oList.attachEvent)
     {
         oList.attachEvent("onclick", onBranchClick);
     }
     // prepares the appearence of the tree
     prepareBranches(oList);
}

/*!
* @function "RemoveSelection()"
* @brief Removes selection of the current item after click on Tree.
*/
function RemoveSelection() 
{
    if (window.getSelection) 
    {  // all browsers, except IE before version 9
        var selection = window.getSelection();                                        
        selection.removeAllRanges();
    }
    else 
    {
        if (document.selection.createRange) 
        {        // Internet Explorer
            var range = document.selection.createRange();
            document.selection.empty();
        }
    }
}

/*!
* @function "onBranchClick(event)"
* @brief Click-event handler.
* @param event Event to be handled.
*/
function onBranchClick(event)
{
     var oBranch, cSubBranches;
     
	 RemoveSelection();
	 
     if (event.target)
     {
         oBranch = event.target;
     }
     else if (event.srcElement)
     {
         oBranch = event.srcElement;
     }
     cSubBranches = oBranch.getElementsByTagName("ul");
     if (cSubBranches.length > 0)
     {
         if (cSubBranches[0].style.display == "block")
         {
             cSubBranches[0].style.display = "none";
             oBranch.style.marginLeft = g_ListMargin;
             oBranch.style.listStyleImage = "url(TreeCollapsed.gif)";
         }
         else
         {
             cSubBranches[0].style.display = "block";
             oBranch.style.marginLeft = g_ListMargin;
             oBranch.style.listStyleImage = "url(TreeExpanded.gif)";
         }
     }
}

/*!
* @function "prepareBranches(oList)"
* @brief Prepares the appearance of the tree.
* @param oList Parent of the tree items.
*/
function prepareBranches(oList)
{
     var cBranches = oList.getElementsByTagName("li");
     var i, n, cSubBranches;
     if (cBranches.length > 0)
     {
         for (i=0, n = cBranches.length; i < n; i++)
         {
             cSubBranches = cBranches[i].getElementsByTagName("ul");
             if (cSubBranches.length > 0)
             {
                 cBranches[i].style.marginLeft = g_ListMargin;
                 prepareBranches(cSubBranches[0]);
                 cSubBranches[0].style.cursor = "auto";
             }
             else
             {
                 cBranches[i].style.marginLeft = g_ListMargin;
                 cBranches[i].style.listStyleImage = "none";
             }
         }
     }
}

/*!******************************************************
* @progsection Toogle expand / collapse all branches Functionality. 
********************************************************/

/*! @variable g_ToogleState Global variable for toogle state. */
var g_ToogleState = 0;

/*!
* @function "initToogleImage()"
* @brief Initializes the toogle image icon.
*/
function initToogleImage()
{
	var indexDiv = document.getElementById("IndexTreeDiv");
	if ( indexDiv != null )
	{
		var node = document.createElement("img");
		node.setAttribute("id", "expandall");
		node.setAttribute("src", "TreeCollapsed.gif");
		node.setAttribute("onClick", "toogleAllBranches();");
		indexDiv.insertBefore(node, indexDiv.firstChild);
	}
}

/*!
* @function "toogleAllBranches()"
* @brief Toogles all branches.
*/
function toogleAllBranches()
{
	var indexDiv = document.getElementById("IndexTreeDiv");
	var icon = document.getElementById("expandall");
	var newState = 1;
	var newIcon = "";
	var dStyle = "";
	
	if ( g_ToogleState == 0 ) 
	{
		newState = 1;
		newIcon = "TreeExpanded.gif";
		dStyle = "block";
	}
	else 
	{
		newState = 0;
		newIcon = "TreeCollapsed.gif";
		dStyle = "none";
	}
	
	var cBranches = indexDiv.getElementsByTagName("ul");
	var n = cBranches.length;
	var cssIcon = "url(" + newIcon + ")";
	
	if ( n > 0 )
	{
		for (var i=0; i < n; i++)
		{
			var curNode = cBranches[i];
			var curID = curNode.id;
			
			if ( typeof curID == 'string' && curID == "" )
			{
				curNode.style.display = dStyle;
			}
			curNode.style.listStyleImage = cssIcon;
			
			// if a branch has been clicked the list item element has been changed, so now set icon correctly.
			var cSubBranches = curNode.getElementsByTagName("li");
			if ( cSubBranches.length > 0 ) 
			{
				var firstLI = cSubBranches[0];
				var firstChilds = firstLI.getElementsByTagName("ul");
				if ( firstChilds.length > 0 ) cSubBranches[0].style.listStyleImage = cssIcon;
			}
		}
	}
	
	icon.src = newIcon;
	g_ToogleState = newState;
}

/*!******************************************************
* @progsection Search Routines.
********************************************************/

/*!
* @variable g_Filelist            Array to be used for files to be searched.
* @variable g_IgnoreCaseFlag      Flag for ignore case.
* @variable g_SearchString        Current search string.
* @variable g_SearchLabel         Label for search.
* @variable g_ResultsLabel        Label for search results.
* @variable g_BusyLabel           Label for busy searching.
* @variable g_CompatErrLabel      Label for busy searching.
* @variable g_IgnoreCaseLabel     Label for ignore case.
*/
var g_Filelist = [];
var g_IgnoreCaseFlag = false;
var g_SearchString = "";
var g_SearchLabel = "Search";
var g_ResultsLabel = "Search results for";
var g_BusyLabel = "Search in progress...";
var g_CompatErrLabel = "Your browser does not support searching.";
var g_IgnoreCaseLabel = "Ignore case";

/*!
* @function "initSearch(searchLabel, resultsLabel, busyLabel, compatErrLabel, ignoreCaseLabel)"
* @brief Init search routines.
* @param searchLabel     Label for search.
* @param resultsLabel    Label for search results.
* @param busyLabel       Label for busy searching.
* @param compatErrLabel  Label for compatibility error.
* @param ignoreCaseLabel Label for ignore case.
*/
function initSearch(searchLabel, resultsLabel, busyLabel, compatErrLabel, ignoreCaseLabel)
{
	if ( searchLabel != null && searchLabel != "" ) g_SearchLabel = searchLabel;
	if ( resultsLabel != null && resultsLabel != "" ) g_ResultsLabel = resultsLabel;
	if ( busyLabel != null && busyLabel != "" ) g_BusyLabel = busyLabel;
	if ( compatErrLabel != null && compatErrLabel != "" ) g_CompatErrLabel = compatErrLabel;
	if ( ignoreCaseLabel != null && ignoreCaseLabel != "" ) g_IgnoreCaseLabel = ignoreCaseLabel;
	
	var searchDiv = document.getElementById("searchbox");
	var aHeadings = searchDiv.getElementsByTagName("label");
	
	if ( isSearchCompatibleBrowser() )
	{
		if ( aHeadings.length > 0 )
		{
			var heading = aHeadings.item(0);
			heading.innerHTML = g_SearchLabel + ":";
		}
		if ( aHeadings.length > 1 )
		{
			var heading = aHeadings.item(1);
			heading.innerHTML = g_IgnoreCaseLabel + ":";
		}
	}
	else
	{
		searchDiv.style.display = "none";
	}
}

/*!
* @brief Checks if browser is compatible with search.
* @return True, if browser is compatible with search.
*/
function isSearchCompatibleBrowser()
{
	var ret = true;
	/*
	if ( (navigator.userAgent.indexOf("Opera") || navigator.userAgent.indexOf('OPR')) != -1 ) ret = false;
	else if(navigator.userAgent.indexOf("Chrome") != -1 ) ret = false;
	else if ( navigator.userAgent.indexOf("Firefox") != -1 ) ret = true;
	else if ( (navigator.userAgent.indexOf("MSIE") != -1 ) || (!!document.documentMode == true ) ) ret = true;
	// else if ( navigator.userAgent.indexOf("Safari") != -1 ) ret = false;
	*/
	return ret;
}

/*!
* @function "getFilename(fullname)"
* @brief Get filename from a full path.
* @param fullname  Full filename path.
* @return Extracted filename.
*/
function getFilename(fullname)
{
	var sResult = fullname;
	var iPos = fullname.lastIndexOf("/");
	if ( iPos >= 0 )
	{
		sResult = fullname.substr(iPos + 1);
	}
	return sResult;
}

// Removes all child elements.
/*!
* @function "removeAllChildren(parent)"
* @brief Removes all child elements.
* @param parent  Parent for which the children have to be removed.
*/
function removeAllChildren(parent)
{
	if ( parent != null ) parent.innerHTML = "";
}

/*!
* @function "appendListItemTextChild(parent, text)"
* @brief Append text child to a li element.
* @param parent  Parent to which the child is to be added.
* @param text    Text of the element.
*/
function appendListItemTextChild(parent, text)
{
	var node = document.createElement("li");
	var textnode = document.createTextNode(text);
	node.appendChild(textnode);
	parent.appendChild(node);
}

/*!
* @function "populateFileList()"
* @brief Populates file list array g_Filelist from the index tree.
* @details Gets the complete li elements. The file names have still to be extracted.
*/
function populateFileList()
{
	g_Filelist = [];
	var indexDiv = document.getElementById("IndexTreeDiv");
	var cBranches = indexDiv.getElementsByTagName("li");
	if ( cBranches.length > 1 )
	{
		// File entry is second entry.
		var fileEntry = cBranches.item(1);
		var cSubBranches = fileEntry.getElementsByTagName("li");
		for (var i=0; i < cSubBranches.length; i++)
		{
			var curElem = cSubBranches.item(i).cloneNode(true);
			var curElemChilds = curElem.getElementsByTagName("ul");
			if ( curElemChilds.length == 0 )
			{
				// Has no ul sub elements, then add.
				g_Filelist.push(curElem);
			}
		}
	}
}

/*!
* @function "finishSearch(resUL)"
* @brief Cleans up after search has finished.
* @param resUL     UL element node containing the results.
* @param bSuccess  Search successful?
*/
function finishSearch(resUL, bSuccess)
{
	var childList = resUL.children;
	if ( childList.length > 0 )
	{
		var elem = childList.item(0);
		if ( bSuccess ) resUL.removeChild(elem);
		else elem.innerHTML = g_CompatErrLabel;
	}
	g_SearchString = "";
}

/*!
* @function "find(sContent, searchString)"
* @brief Find routine of the search.
* @param sContent     Content in which to search.
* @param searchString String to be searched for.
* @return True, if found.
*/
function find(sContent, searchString)
{
	var bResult = false;
	var iFound = sContent.indexOf(searchString);
	if ( iFound >= 0 ) bResult = true;
	return bResult;
}

/*!
* @function "submitsearch()"
* @brief Handler for the submit event.
*/
function submitsearch()
{
	var searchStrForm = document.getElementById("searchstr");
	var searchIgnoreForm = document.getElementById("ignoreflag");
	g_SearchString = searchStrForm.value;
	g_IgnoreCaseFlag = searchIgnoreForm.checked;
	
	var resHead = document.getElementById("searchhead");
	var resUL = document.getElementById("searchresults");
	var myIframe = document.getElementById("searchtmpfrm");
	
	// Clean previous results.
	removeAllChildren(resHead);
	removeAllChildren(resUL);
	
	if ( g_SearchString != "" )
	{
		// Initialize result list and get file list for processing.
		var textnode = document.createTextNode(g_ResultsLabel + ' "' + g_SearchString + '":');
		resHead.appendChild(textnode);
		appendListItemTextChild(resUL, g_BusyLabel);
		populateFileList();
	
		// Get first file name and load. To be continued in onIFrameLoaded.
		for (var i=0; i < g_Filelist.length; i++)
		{
			var curElem = g_Filelist[i];
			var curLinks = curElem.getElementsByTagName("a");
			var sFilename = "";
			if ( curLinks.length > 0 )
			{
				sFilename = getFilename(curLinks.item(0).href);

				var sID = "Src_" + sFilename;
				try
				{
					var doc = myIframe.contentDocument || myIframe.contentWindow.document;
					var elemFileEntry = doc.getElementById(sID);
					sContent = elemFileEntry.innerHTML;
				}
				catch (err)
				{
					// Not supported by browser. Abort search.
					g_Filelist = [];
					finishSearch(resUL, false);
					return;
				}
		
				var found = false;
				if ( g_IgnoreCaseFlag )
					found = find(sContent.toUpperCase(), g_SearchString.toUpperCase());
				else
					found = find(sContent, g_SearchString);
				
				if ( found )
				{
					resUL.appendChild(curElem);
				}
			}
		}
			
		finishSearch(resUL, true);
	}
}

/*!
* @function "checkReturn(e)"
* @brief Handler for the submit event.
* @param e  Event object.
* @returns Flag, if event should be propagated further.
*/
function checkReturn(e)
{
	if ( e.which == 13 || e.keyCode == 13 ) 
	{
		submitsearch();
		return false;
	}
	return true;
}
